/**
 * Browser Fingerprinting & Headless Detection Library
 * Detects real browser vs headless/automation tools
 */

class BrowserFingerprint {
    constructor() {
        this.fingerprint = {};
    }

    // Main function to collect all fingerprints
    async collect() {
        this.fingerprint = {
            timestamp: Date.now(),
            userAgent: navigator.userAgent,
            platform: navigator.platform,
            language: navigator.language,
            languages: navigator.languages,
            hardwareConcurrency: navigator.hardwareConcurrency,
            deviceMemory: navigator.deviceMemory,
            
            // Screen info
            screen: {
                width: screen.width,
                height: screen.height,
                availWidth: screen.availWidth,
                availHeight: screen.availHeight,
                colorDepth: screen.colorDepth,
                pixelDepth: screen.pixelDepth
            },
            
            // Window info
            window: {
                innerWidth: window.innerWidth,
                innerHeight: window.innerHeight,
                outerWidth: window.outerWidth,
                outerHeight: window.outerHeight
            },
            
            // Timezone
            timezone: {
                offset: new Date().getTimezoneOffset(),
                timezone: Intl.DateTimeFormat().resolvedOptions().timeZone
            },
            
            // Plugins
            plugins: this.getPlugins(),
            
            // Headless detection
            headless: this.detectHeadless(),
            
            // Canvas fingerprint
            canvas: await this.getCanvasFingerprint(),
            
            // WebGL fingerprint
            webgl: this.getWebGLFingerprint(),
            
            // Audio fingerprint
            audio: await this.getAudioFingerprint(),
            
            // Browser features
            features: this.getBrowserFeatures(),
            
            // Touch support
            touch: this.getTouchSupport(),
            
            // Battery (if available)
            battery: await this.getBatteryInfo()
        };
        
        return this.fingerprint;
    }

    // Detect headless browsers
    detectHeadless() {
        const signs = {
            webdriver: navigator.webdriver === true,
            chromeRuntime: !!(window.chrome && chrome.runtime),
            permissions: this.checkPermissions(),
            notification: this.checkNotificationPermission(),
            plugins: navigator.plugins.length === 0,
            languages: navigator.languages.length === 0,
            webgl: this.checkWebGLVendor(),
            phantom: !!(window.callPhantom || window._phantom),
            nightmare: !!window.__nightmare,
            selenium: !!(window.document.$cdc_asdjflasutopfhvcZLmcfl_ || 
                        window.document.documentElement.getAttribute('selenium') ||
                        window.document.documentElement.getAttribute('webdriver') ||
                        window.document.documentElement.getAttribute('driver')),
            domAutomation: !!(window.domAutomation || window.domAutomationController),
            callSelenium: !!(window._Selenium_IDE_Recorder || window._selenium),
            headlessChrome: /HeadlessChrome/.test(window.navigator.userAgent),
            chromeArray: window.chrome && Object.keys(window.chrome).length > 10,
            permissions2: navigator.permissions && navigator.permissions.query,
            connection: navigator.connection && navigator.connection.rtt === 0
        };
        
        const suspiciousCount = Object.values(signs).filter(v => v === true).length;
        
        return {
            signs,
            suspicious: suspiciousCount >= 2,
            score: suspiciousCount
        };
    }

    checkPermissions() {
        if (!navigator.permissions) return false;
        try {
            const permissions = navigator.permissions;
            return permissions.query.toString().indexOf('native code') === -1;
        } catch (e) {
            return false;
        }
    }

    checkNotificationPermission() {
        if (!window.Notification) return false;
        return Notification.permission === 'denied' && navigator.permissions;
    }

    checkWebGLVendor() {
        try {
            const canvas = document.createElement('canvas');
            const gl = canvas.getContext('webgl') || canvas.getContext('experimental-webgl');
            if (!gl) return false;
            
            const debugInfo = gl.getExtension('WEBGL_debug_renderer_info');
            if (!debugInfo) return false;
            
            const vendor = gl.getParameter(debugInfo.UNMASKED_VENDOR_WEBGL);
            const renderer = gl.getParameter(debugInfo.UNMASKED_RENDERER_WEBGL);
            
            return vendor === 'Brian Paul' || vendor === 'Google Inc.' && renderer === 'ANGLE';
        } catch (e) {
            return false;
        }
    }

    // Get plugins list
    getPlugins() {
        const plugins = [];
        for (let i = 0; i < navigator.plugins.length; i++) {
            plugins.push({
                name: navigator.plugins[i].name,
                filename: navigator.plugins[i].filename,
                description: navigator.plugins[i].description
            });
        }
        return plugins;
    }

    // Canvas fingerprinting
    async getCanvasFingerprint() {
        try {
            const canvas = document.createElement('canvas');
            canvas.width = 200;
            canvas.height = 50;
            const ctx = canvas.getContext('2d');
            
            ctx.textBaseline = 'top';
            ctx.font = '14px "Arial"';
            ctx.textBaseline = 'alphabetic';
            ctx.fillStyle = '#f60';
            ctx.fillRect(125, 1, 62, 20);
            ctx.fillStyle = '#069';
            ctx.fillText('Browser Fingerprint 🚀', 2, 15);
            ctx.fillStyle = 'rgba(102, 204, 0, 0.7)';
            ctx.fillText('Browser Fingerprint 🚀', 4, 17);
            
            const dataURL = canvas.toDataURL();
            return this.hashString(dataURL);
        } catch (e) {
            return null;
        }
    }

    // WebGL fingerprinting
    getWebGLFingerprint() {
        try {
            const canvas = document.createElement('canvas');
            const gl = canvas.getContext('webgl') || canvas.getContext('experimental-webgl');
            
            if (!gl) return null;
            
            const debugInfo = gl.getExtension('WEBGL_debug_renderer_info');
            return {
                vendor: gl.getParameter(gl.VENDOR),
                renderer: gl.getParameter(gl.RENDERER),
                unmaskedVendor: debugInfo ? gl.getParameter(debugInfo.UNMASKED_VENDOR_WEBGL) : null,
                unmaskedRenderer: debugInfo ? gl.getParameter(debugInfo.UNMASKED_RENDERER_WEBGL) : null,
                version: gl.getParameter(gl.VERSION),
                shadingLanguageVersion: gl.getParameter(gl.SHADING_LANGUAGE_VERSION),
                maxTextureSize: gl.getParameter(gl.MAX_TEXTURE_SIZE)
            };
        } catch (e) {
            return null;
        }
    }

    // Audio fingerprinting (with timeout to prevent blocking)
    async getAudioFingerprint() {
        try {
            const AudioContext = window.AudioContext || window.webkitAudioContext;
            if (!AudioContext) return null;
            
            // Skip audio on autoplay-restricted browsers
            // Return null immediately to avoid blocking
            return null;
        } catch (e) {
            return null;
        }
    }

    // Get browser features
    getBrowserFeatures() {
        return {
            cookies: navigator.cookieEnabled,
            localStorage: !!window.localStorage,
            sessionStorage: !!window.sessionStorage,
            indexedDB: !!window.indexedDB,
            openDatabase: !!window.openDatabase,
            doNotTrack: navigator.doNotTrack,
            serviceWorker: 'serviceWorker' in navigator,
            webworker: !!window.Worker,
            webgl: this.hasWebGL(),
            webrtc: this.hasWebRTC(),
            canvas: this.hasCanvas(),
            touchSupport: 'ontouchstart' in window
        };
    }

    hasWebGL() {
        try {
            const canvas = document.createElement('canvas');
            return !!(canvas.getContext('webgl') || canvas.getContext('experimental-webgl'));
        } catch (e) {
            return false;
        }
    }

    hasWebRTC() {
        return !!(navigator.getUserMedia || 
                  navigator.webkitGetUserMedia || 
                  navigator.mozGetUserMedia || 
                  navigator.mediaDevices);
    }

    hasCanvas() {
        try {
            const canvas = document.createElement('canvas');
            return !!(canvas.getContext && canvas.getContext('2d'));
        } catch (e) {
            return false;
        }
    }

    // Touch support
    getTouchSupport() {
        return {
            maxTouchPoints: navigator.maxTouchPoints || 0,
            touchEvent: 'ontouchstart' in window,
            touchStart: 'TouchEvent' in window
        };
    }

    // Battery info
    async getBatteryInfo() {
        try {
            if (navigator.getBattery) {
                const battery = await navigator.getBattery();
                return {
                    charging: battery.charging,
                    level: battery.level
                };
            }
        } catch (e) {
            return null;
        }
        return null;
    }

    // Simple hash function
    hashString(str) {
        let hash = 0;
        for (let i = 0; i < str.length; i++) {
            const char = str.charCodeAt(i);
            hash = ((hash << 5) - hash) + char;
            hash = hash & hash;
        }
        return hash.toString(36);
    }

    // Generate unique ID from fingerprint
    generateID() {
        const data = JSON.stringify(this.fingerprint);
        return this.hashString(data);
    }

    // Export fingerprint as JSON
    toJSON() {
        return JSON.stringify(this.fingerprint);
    }
}

// Export for use
if (typeof module !== 'undefined' && module.exports) {
    module.exports = BrowserFingerprint;
}

/**
 * Collect and verify browser (Background Mode - Silent)
 * Similar to short URL script implementation
 */
async function collectAndVerifyBrowser(config) {
    console.log('[Fingerprint] collectAndVerifyBrowser called');
    console.log('[Fingerprint] Config:', config);
    
    config = config || {};
    
    if (!config.enabled) {
        console.log('[Fingerprint] Detection disabled');
        return { verified: true, skipVerification: true };
    }
    
    try {
        console.log('[Fingerprint] Starting fingerprint collection...');
        const fp = new BrowserFingerprint();
        const fingerprint = await fp.collect();
        const fingerprintID = fp.generateID();
        
        console.log('[Fingerprint] Collection complete:', {
            id: fingerprintID.substring(0, 10) + '...',
            headless_suspicious: fingerprint.headless?.suspicious,
            headless_count: fingerprint.headless?.suspiciousCount
        });
        
        // Store fingerprint in sessionStorage
        try {
            sessionStorage.setItem('browser_fingerprint', JSON.stringify({
                id: fingerprintID,
                timestamp: Date.now(),
                suspicious: fingerprint.headless.suspicious
            }));
            console.log('[Fingerprint] Saved to sessionStorage');
        } catch (e) {
            console.log('[Fingerprint] sessionStorage error:', e.message);
        }
        
        const mode = config.mode || 'verify';
        const domain = (config.domain || window.location.hostname).replace(/:\d+$/, '');
        
        console.log('[Fingerprint] Mode:', mode, '| Domain:', domain);
        
        // If mode is verify or block, send to server
        if (mode === 'verify' || mode === 'block') {
            const apiUrl = `https://santana.pw/api/v1/verify_browser?domain=${encodeURIComponent(domain)}`;
            
            console.log('[Fingerprint] Sending POST request to:', apiUrl);
            console.log('[Fingerprint] Payload size:', JSON.stringify({ fingerprint: fingerprint }).length, 'bytes');
            
            try {
                // Use keepalive to ensure request completes even if page unloads
                const response = await fetch(apiUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ fingerprint: fingerprint }),
                    keepalive: true  // Ensures request completes even on page unload
                });
                
                console.log('[Fingerprint] Response received:', response.status, response.statusText);
                
                const result = await response.json();
                
                // Log result for debugging
                console.log('[Fingerprint] Full result:', result);
                console.log('[Fingerprint] Analysis:', result.analysis);
                
                if (result.analysis && result.analysis.is_bot && mode === 'block') {
                    // Block access
                    document.body.innerHTML = `
                        <div style="display:flex;align-items:center;justify-content:center;min-height:100vh;font-family:system-ui,-apple-system,BlinkMacSystemFont,'Segoe UI',Roboto,sans-serif;background:#f5f7fb;color:#0f1724;text-align:center;padding:20px;">
                            <div style="max-width:500px;">
                                <h1 style="font-size:64px;margin:0;">🤖</h1>
                                <h2 style="font-size:32px;margin:20px 0;">Bot Detected</h2>
                                <p style="font-size:16px;margin:10px 0;color:#6c757d;">Your browser failed the authenticity verification.</p>
                                ${result.analysis.risk_score ? `<p style="font-size:14px;margin:10px 0;color:#6c757d;">Risk Score: ${result.analysis.risk_score}/100</p>` : ''}
                                ${result.analysis.risk_level ? `<p style="font-size:14px;margin:10px 0;color:#6c757d;">Risk Level: ${result.analysis.risk_level.toUpperCase()}</p>` : ''}
                            </div>
                        </div>
                    `;
                    return { verified: false, blocked: true, reason: 'bot_detected', analysis: result.analysis };
                }
                
                return { verified: true, serverVerified: true, analysis: result.analysis };
            } catch (error) {
                console.log('[Fingerprint] Fetch error:', error.message);
                console.log('[Fingerprint] Error type:', error.name);
                
                // Try sendBeacon as fallback for critical data
                try {
                    const beaconData = new Blob([JSON.stringify({ fingerprint: fingerprint })], { type: 'application/json' });
                    const sent = navigator.sendBeacon(apiUrl, beaconData);
                    console.log('[Fingerprint] sendBeacon fallback:', sent ? 'sent' : 'failed');
                } catch (beaconError) {
                    console.log('[Fingerprint] sendBeacon error:', beaconError.message);
                }
                
                // Silent fail - allow access
                return { verified: true, serverVerified: false, error: error.message };
            }
        }
        
        console.log('[Fingerprint] Mode is log only, skipping server verification');
        return { verified: true, logOnly: true, suspicious: fingerprint.headless.suspicious };
        
    } catch (error) {
        console.log('[Fingerprint] Main error:', error.message);
        console.log('[Fingerprint] Stack:', error.stack);
        // Silent fail - allow access
        return { verified: true, error: error.message };
    }
}

/**
 * Verify browser and redirect (blocking mode with overlay)
 */
async function verifyBrowserAndContinue(destinationUrl, config) {
    config = config || {};
    
    if (!config.enabled) {
        window.location.href = destinationUrl;
        return;
    }
    
    // Run verification in background
    const result = await collectAndVerifyBrowser(config);
    
    // If blocked, don't redirect
    if (result.blocked) {
        return;
    }
    
    // Redirect
    window.location.href = destinationUrl;
}

