(async () => {
  // Prevent multiple executions of decrypt
  if (window.__DECRYPT_RUNNING__) {
    return;
  }
  window.__DECRYPT_RUNNING__ = true;
  
  // Prefer reading Base64 from the script tag attribute to avoid inline-script CSP issues
  const currentScript = document.currentScript || document.querySelector('script[src*="/assets/encrypt.js"]');
  const b64 = (currentScript && currentScript.getAttribute('data-html-b64')) || (window.__HTML_B64__ || '');

  // If Web Crypto isn't available (e.g., non-HTTPS), render directly to avoid a blank page
  // If no payload, render nothing (feature disabled)
  if (!b64) return;

  if (!('crypto' in window) || !window.crypto.subtle || !window.isSecureContext) {
    const html = b64 ? atob(b64) : '';
    // Use DOMParser to avoid CSP issues
    const parser = new DOMParser();
    const doc = parser.parseFromString(html, 'text/html');
    const pageTitle = doc.title;
    const loadingOverlay = document.getElementById('loading-overlay');
    
    document.documentElement.innerHTML = doc.documentElement.innerHTML;
    if (pageTitle) {
      document.title = pageTitle;
    }
    
    // Restore loading overlay
    if (loadingOverlay && document.body) {
      document.body.appendChild(loadingOverlay);
    }
    
    // Re-execute inline scripts
    // Skip if scripts already executed to prevent duplicate event listeners
    if (!window.__SCRIPTS_EXECUTED__) {
      window.__SCRIPTS_EXECUTED__ = true;
      const scripts = document.querySelectorAll('script');
      scripts.forEach(oldScript => {
        if (oldScript.src) return;
        try {
          const newScript = document.createElement('script');
          Array.from(oldScript.attributes).forEach(attr => {
            newScript.setAttribute(attr.name, attr.value);
          });
          newScript.textContent = oldScript.textContent;
          if (oldScript.parentNode) {
            oldScript.parentNode.replaceChild(newScript, oldScript);
          }
        } catch (err) {
          console.warn('Failed to re-execute script:', err);
          // Script will still be in DOM, just not re-executed
        }
      });
    }
    // Remove loading overlay with fade-out effect
    setTimeout(() => {
      const overlay = document.getElementById('loading-overlay');
      if (overlay) {
        overlay.classList.add('fade-out');
        setTimeout(() => overlay.remove(), 200);
      }
      // Only dispatch events if not already dispatched
      if (!window.__EVENTS_DISPATCHED__) {
        window.__EVENTS_DISPATCHED__ = true;
        try { document.dispatchEvent(new Event('DOMContentLoaded')); } catch (e) {}
        try { window.dispatchEvent(new Event('load')); } catch (e) {}
      }
    }, 0);
    return;
  }

  const b64ToBytes = (b64str) => Uint8Array.from(atob(b64str), c => c.charCodeAt(0));
  const dec = new TextDecoder();

  const plaintext = b64ToBytes(b64);

  const keyMaterial = crypto.getRandomValues(new Uint8Array(32));
  const key = await crypto.subtle.importKey('raw', keyMaterial, { name: 'AES-GCM' }, false, ['encrypt','decrypt']);
  const iv = crypto.getRandomValues(new Uint8Array(12));

  const cipherBuf = await crypto.subtle.encrypt({ name: 'AES-GCM', iv }, key, plaintext);
  const cipherBytes = new Uint8Array(cipherBuf);

  const importedKey = await crypto.subtle.importKey('raw', keyMaterial, { name: 'AES-GCM' }, false, ['decrypt']);
  const decrypted = await crypto.subtle.decrypt({ name: 'AES-GCM', iv }, importedKey, cipherBytes);
  const html = dec.decode(new Uint8Array(decrypted));

  // Use DOMParser instead of document.write to avoid CSP issues
  const parser = new DOMParser();
  const doc = parser.parseFromString(html, 'text/html');
  
  // Extract and save title before replacing
  const pageTitle = doc.title;
  
  // Save loading overlay before replacing
  const loadingOverlay = document.getElementById('loading-overlay');
  
  // Replace current document content
  document.documentElement.innerHTML = doc.documentElement.innerHTML;
  
  // Restore title (innerHTML doesn't preserve document.title)
  if (pageTitle) {
    document.title = pageTitle;
  }
  
  // Restore loading overlay
  if (loadingOverlay && document.body) {
    document.body.appendChild(loadingOverlay);
  }
  
  // Re-execute script tags (innerHTML doesn't execute scripts)
  // Skip if scripts already executed to prevent duplicate event listeners
  if (!window.__SCRIPTS_EXECUTED__) {
    window.__SCRIPTS_EXECUTED__ = true;
    const scripts = document.querySelectorAll('script');
    scripts.forEach(oldScript => {
      if (oldScript.src) return; // Skip external scripts (already loaded)
      try {
        const newScript = document.createElement('script');
        Array.from(oldScript.attributes).forEach(attr => {
          newScript.setAttribute(attr.name, attr.value);
        });
        newScript.textContent = oldScript.textContent;
        if (oldScript.parentNode) {
          oldScript.parentNode.replaceChild(newScript, oldScript);
        }
      } catch (err) {
        console.warn('Failed to re-execute script:', err);
        // Script will still be in DOM, just not re-executed
      }
    });
  }
  
  // Remove loading overlay with fade-out effect
  setTimeout(() => {
    const overlay = document.getElementById('loading-overlay');
    if (overlay) {
      overlay.classList.add('fade-out');
      setTimeout(() => overlay.remove(), 200);
    }
    // Only dispatch events if not already dispatched
    if (!window.__EVENTS_DISPATCHED__) {
      window.__EVENTS_DISPATCHED__ = true;
      try { document.dispatchEvent(new Event('DOMContentLoaded')); } catch (e) {}
      try { window.dispatchEvent(new Event('load')); } catch (e) {}
    }
  }, 0);
})().catch(err => {
  console.error(err);
  // Remove loading overlay
  const overlay = document.getElementById('loading-overlay');
  if (overlay) overlay.remove();
  
  const errorHtml = '<!DOCTYPE html><html><head><meta charset="utf-8"><title>Error</title></head><body style="display:flex;align-items:center;justify-content:center;height:100vh;margin:0;font-family:system-ui;background:#f5f7fb;color:#0f1724;text-align:center;"><div><div style="font-size:48px;margin-bottom:20px;">⚠️</div><div style="font-size:20px;font-weight:600;">Decryption Error</div><div style="margin-top:10px;opacity:0.7;">Please refresh the page.</div></div></body></html>';
  const parser = new DOMParser();
  const doc = parser.parseFromString(errorHtml, 'text/html');
  document.documentElement.innerHTML = doc.documentElement.innerHTML;
  document.title = 'Error';
});


