<?php
if (!defined('ALLOW')) { http_response_code(403); exit; }

/**
 * Get application settings from external API (NO CACHE)
 * API: https://santana.pw/api/v1/settings
 * Direct API call every request for real-time updates
 */
function get_app_config(): array {
    static $cachedConfig = null;
    
    // Return memory cached config if already loaded in this request
    if ($cachedConfig !== null) {
        return $cachedConfig;
    }
    
    // Get domain and remove port if present
    $fullHost = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $domain = preg_replace('/:\d+$/', '', $fullHost);
    
    // Fetch directly from API - no file cache
    $apiUrl = 'https://santana.pw/api/v1/settings';
    $params = http_build_query(['domain' => $domain]);
    $url = $apiUrl . '?' . $params;
    
    // Initialize cURL with shorter timeout
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 3); // Reduced to 3 seconds
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 2); // Connection timeout 2 seconds
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (compatible; PHP-App-Client/1.0)');
    
    $headers = [
        'Accept: application/json',
        'Cache-Control: no-cache'
    ];
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    $curlInfo = curl_getinfo($ch);
    curl_close($ch);
    
    // Check for cURL errors or non-200 response
    if ($response === false || $httpCode !== 200) {
        $error = $curlError ?: 'HTTP ' . $httpCode;
        $apiResponse = json_decode($response ?: '{}', true);
        $apiErrorMsg = isset($apiResponse['error']) ? $apiResponse['error'] : 'Unknown error';
        
        $debugInfo = [
            'url' => $url,
            'domain' => $domain,
            'http_code' => $httpCode,
            'curl_error' => $curlError,
            'api_error' => $apiErrorMsg,
            'response_preview' => substr($response ?: '', 0, 200),
            'curl_info' => [
                'effective_url' => $curlInfo['url'] ?? '',
                'total_time' => $curlInfo['total_time'] ?? 0,
                'connect_time' => $curlInfo['connect_time'] ?? 0,
            ]
        ];
        error_log("Failed to fetch config from API: " . $error . " | Debug: " . json_encode($debugInfo));
        
        // Build detailed user-friendly error message
        $errorMessage = "⚠️ CONFIGURATION API ERROR\n\n";
        $errorMessage .= "Status: HTTP " . $httpCode . "\n";
        $errorMessage .= "API Response: " . $apiErrorMsg . "\n";
        $errorMessage .= "Domain: " . $domain . "\n\n";
        
        // Add specific instructions based on error type
        if ($httpCode == 403) {
            $errorMessage .= "🔴 PROBLEM:\n";
            $errorMessage .= "Your domain '" . $domain . "' is not authorized by the API.\n\n";
            $errorMessage .= "❌ ACCESS DENIED\n";
            $errorMessage .= "This application requires API authorization.\n\n";
            $errorMessage .= "📧 CONTACT:\n";
            $errorMessage .= "Register your domain at santana.pw to get access.\n";
        } elseif ($httpCode == 401) {
            $errorMessage .= "🔴 PROBLEM:\n";
            $errorMessage .= "API authentication required.\n\n";
            $errorMessage .= "❌ ACCESS DENIED\n";
            $errorMessage .= "Valid API credentials required.\n\n";
            $errorMessage .= "📧 CONTACT:\n";
            $errorMessage .= "Contact santana.pw for API credentials.\n";
        } else {
            $errorMessage .= "🔴 PROBLEM:\n";
            $errorMessage .= "Failed to connect to configuration API.\n\n";
            $errorMessage .= "⚠️ TEMPORARY ERROR\n";
            $errorMessage .= "The API service may be temporarily unavailable.\n\n";
            $errorMessage .= "🔄 TRY AGAIN:\n";
            $errorMessage .= "Please refresh the page in a few moments.\n";
        }
        
        throw new Exception($errorMessage);
    }
    
    // Parse JSON response
    $data = json_decode($response, true);
    
    if (!is_array($data) || !isset($data['success']) || $data['success'] !== true) {
        error_log("Invalid config API response: " . substr($response, 0, 200));
        throw new Exception("Invalid configuration API response");
    }
    
    if (!isset($data['settings']) || !is_array($data['settings'])) {
        throw new Exception("Configuration settings not found in API response");
    }
    
    // Cache in memory for this request only (no file cache)
    $cachedConfig = $data['settings'];
    
    return $cachedConfig;
}

/**
 * Get a specific config value
 * 
 * @param string $key The config key (supports dot notation for nested values)
 * @param mixed $default Default value if key not found (should not be used per user requirement)
 * @return mixed
 */
function get_config(string $key, $default = null) {
    $config = get_app_config();
    
    // Support dot notation for nested keys (e.g., 'flow_settings.security_mmn_enabled')
    if (strpos($key, '.') !== false) {
        $keys = explode('.', $key);
        $value = $config;
        
        foreach ($keys as $k) {
            if (!isset($value[$k])) {
                if ($default === null) {
                    throw new Exception("Configuration key not found: " . $key);
                }
                return $default;
            }
            $value = $value[$k];
        }
        
        return $value;
    }
    
    // Simple key lookup
    if (!isset($config[$key])) {
        if ($default === null) {
            throw new Exception("Configuration key not found: " . $key);
        }
        return $default;
    }
    
    return $config[$key];
}

