<?php
if (!defined('ALLOW')) { http_response_code(403); exit; }

/**
 * ========================================
 * EXTERNAL API BLOCKER ONLY
 * No local rules - 100% API-based blocking
 * ========================================
 */

/**
 * Check visitor using external API blocker (NO CACHE)
 * API: https://santana.pw/api/v1/blocker
 * Direct API call every request for real-time blocking
 */
function check_external_blocker(string $ip, string $userAgent, string $domain): array {
    // Fetch directly from API - no cache
    $apiUrl = 'https://santana.pw/api/v1/blocker';
    $params = http_build_query([
        'ip' => $ip,
        'ua' => $userAgent,
        'domain' => $domain
    ]);
    $url = $apiUrl . '?' . $params;
    
    // Initialize cURL with shorter timeout
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 2); // Reduced to 2 seconds
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 1); // Connection timeout 1 second
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Faster without SSL verify
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    // Check for cURL errors
    if ($response === false || $httpCode !== 200) {
            return [
            'success' => false,
            'error' => $curlError ?: 'HTTP ' . $httpCode,
            'action' => 'allow' // Default to allow on API failure
        ];
    }
    
    // Parse JSON response
    $data = json_decode($response, true);
    
    if (!is_array($data) || !isset($data['action'])) {
        return [
            'success' => false,
            'error' => 'Invalid API response',
            'action' => 'allow' // Default to allow on invalid response
        ];
    }
    
    // Return result directly - no caching
    return array_merge(['success' => true], $data);
}

function is_blocked(): array {
    $ip = get_client_ip();
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    $fullHost = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $domain = preg_replace('/:\d+$/', '', $fullHost); // Remove port
    
    // Check using external API blocker
    $apiResult = check_external_blocker($ip, $userAgent, $domain);
    
    // Only use external API - no local fallback
    if ($apiResult['success'] && isset($apiResult['action'])) {
        // API call successful
        if ($apiResult['action'] === 'block') {
            // Extract blocking reason
            $reason = 'Access blocked by security system';
            $details = [];
            
            if (isset($apiResult['risk_score']['reasons']) && !empty($apiResult['risk_score']['reasons'])) {
                $details = $apiResult['risk_score']['reasons'];
                $reason = implode(', ', $details);
            }
            
            // Check specific blocking reasons
            if (isset($apiResult['ip_details'])) {
                $ipDetails = $apiResult['ip_details'];
                if (!empty($ipDetails['proxy']) || !empty($ipDetails['hosting'])) {
                    $reason = 'Proxy/VPN/Hosting blocked';
                }
            }
            
            if (isset($apiResult['ua_details']['is_bot']) && $apiResult['ua_details']['is_bot']) {
                $reason = 'Bot detected';
            }
            
            return [
                'blocked' => true,
                'reason' => $reason,
                'value' => $ip,
                'details' => $details,
                'source' => 'external_api',
                'api_data' => $apiResult
            ];
        }
    }
    
    // API returned allow or failed - allow access
        return [
        'blocked' => false,
        'api_data' => $apiResult
    ];
}

function render_blocked_page(array $blockInfo): void {
    // Standard blocked page for all types of blocks (including blacklisted IPs)
    $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Access Denied</title>
    <style>
        body { 
            font-family: -apple-system, BlinkMacSystemFont, sans-serif; 
            background: #f8f9fa; 
            margin: 0; 
            display: flex; 
            align-items: center; 
            justify-content: center; 
            min-height: 100vh; 
            color: #495057;
        }
        .blocked-container { 
            background: white; 
            padding: 60px 40px; 
            box-shadow: 0 10px 30px rgba(0,0,0,0.1); 
            text-align: center; 
            max-width: 500px;
        }
        .blocked-icon { 
            font-size: 64px; 
            color: #dc3545; 
            margin-bottom: 20px; 
        }
        h1 { 
            color: #dc3545; 
            margin-bottom: 20px; 
            font-size: 28px;
        }
        p { 
            color: #6c757d; 
            margin-bottom: 10px; 
            line-height: 1.5;
        }
    </style>
</head>
<body>
    <div class="blocked-container">
        <div class="blocked-icon">🚫</div>
        <h1>Access Denied</h1>
        <p>Your access to this site has been blocked.</p>
        <p>If you believe this is an error, please contact the administrator.</p>
    </div>
</body>
</html>';
    
    http_response_code(403);
    header('Content-Type: text/html; charset=UTF-8');
    echo $html;
    exit;
}
